﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.IO;
using System.Net;
using EPiServer.Forms.Internal.GeoData;
using EPiServer.Logging;
using EPiServer.Personalization;
using EPiServer.Personalization.Internal;
using EPiServer.Web;
using MaxMind.Db;
using MaxMind.GeoIP2;

namespace AlloySample.Internal.GeoData
{
    /// <summary>
    /// Implementation of <see cref="IGeolocationProvider"/> for MaxMind Geolocation databases based on the .mmdb format.
    /// </summary>
    /// <remarks>
    /// This provider have currently only been verified with MaxMind GeoLite2 Country and City geo databases.
    /// </remarks>
    public class MaxMindGeoLocationCustomProvider : GeolocationProviderBase, ICustomGeolocationProvider
    {
        internal const string DatabasePathConfigKey = "databaseFileName";
        internal const string LocationsDatabasePathConfigKey = "locationsFileName";
        private static readonly ILogger _logger = LogManager.GetLogger();
        private Capabilities? _capabilities;
        private readonly MaxMindGeolocationOptions _options;
        private Lazy<string> _databasePath;
        private Lazy<CsvGeographicLocationsProvider> _locationsProvider;
        private readonly IWebHostingEnvironment _webHostingEnvironment;

        /// <summary>
        /// Initializes an instance of <see cref="MaxMindGeolocationProvider"/>.
        /// </summary>
        public MaxMindGeoLocationCustomProvider(MaxMindGeolocationOptions options, IWebHostingEnvironment webHostingEnvironment)
        {
            _options = options ?? throw new ArgumentNullException(nameof(options));
            _webHostingEnvironment = webHostingEnvironment ?? throw new ArgumentNullException(nameof(webHostingEnvironment));

            InitializeDatabasePath(options.DatabasePath);
            InitializeLocationsProvider(options.LocationsDatabasePath);
        }

        /// <inheritdoc />
        public override Capabilities Capabilities
        {
            get
            {
                if (_capabilities == null)
                {
                    if (string.IsNullOrEmpty(DatabasePath))
                    {
                        _capabilities = 0;
                    }
                    else
                    {
                        _capabilities = _options.Capabilities ?? GetCapabilitiesFromKnownDatabaseType(DatabasePath);
                    }
                }
                return _capabilities ?? 0;
            }
        }

        // Exposed for unit testing reasons
        internal string DatabasePath => _databasePath.Value;

        // Exposed for unit testing reasons
        internal CsvGeographicLocationsProvider LocationsProvider => _locationsProvider.Value;

        public IGeoDataProcessService GeoDataProcessService
        {
            get
            {
                return new CustomGeoDataProcessService();
            }

        }

        /// <summary>
        /// Inititalize provider with a collection of configuration parameters.
        /// </summary>
        /// <remarks>
        /// This method exists for the single purpose of supporting web.config driven initialization.
        /// </remarks>
        public override void Initialize(string name, NameValueCollection config)
        {
            if (config == null)
                throw new ArgumentNullException(nameof(config));

            if (_databasePath.IsValueCreated)
            {
                _capabilities = null;
            }

            InitializeDatabasePath(config[DatabasePathConfigKey]);
            InitializeLocationsProvider(config[LocationsDatabasePathConfigKey]);

            base.Initialize(name, config);
        }

        /// <inheritdoc />
        public override IEnumerable<string> GetContinentCodes()
            => LocationsProvider.GetContinentCodes();

        /// <inheritdoc />
        public override IEnumerable<string> GetCountryCodes(string continentCode)
            => LocationsProvider.GetCountryCodes(continentCode);

        /// <inheritdoc />
        public override IEnumerable<string> GetRegions(string countryCode)
            => LocationsProvider.GetRegions(countryCode);

        /// <inheritdoc />
        public override IGeolocationResult Lookup(IPAddress address)
        {
            if (address == null)
                throw new ArgumentNullException(nameof(address));

            if (IPAddress.IsLoopback(address)
                || address == IPAddress.Any
                || address == IPAddress.Broadcast
                || address == IPAddress.None
                || address == IPAddress.IPv6Any
                || address == IPAddress.IPv6None)
            {
                return null;
            }

            if (string.IsNullOrEmpty(DatabasePath))
            {
                return null;
            }

            using (var reader = new DatabaseReader(DatabasePath))
            {
                var result = new EPiServer.Forms.Internal.GeoData.GeolocationResult();
                if (UseCityRequest(reader))
                {
                    if (reader.TryCity(address, out var response))
                    {
                        //get whatever you want heres
                        result.city = response.City?.Name??"";
                        result.ip = response.Traits.IPAddress;
                        result.country_code = response.Country?.IsoCode??"";
                        result.country_name = response.Country?.Name ?? "";
                        result.region_code = response.Continent?.Code??"";
                        result.region_name = response.Continent?.Name??"";
                        result.zip_code = response.Postal?.Code ?? "";
                        result.ContinentCode = response.Continent.Code;
                        result.CountryCode = response.Country.IsoCode;
                        result.time_zone = response.Location.TimeZone;
                       
                        
                    }
                }
                else
                {
                    if (reader.TryCountry(address, out var response))
                    {

                        result.ip = response.Traits.IPAddress;
                        result.country_code = response.Country?.IsoCode ?? "";
                        result.country_name = response.Country?.Name ?? "";
                        result.region_code = response.Continent?.Code ?? "";
                        result.region_name = response.Continent?.Name ?? "";
                        result.ContinentCode = response.Continent.Code;
                        result.CountryCode = response.Country.IsoCode;
                    }
                }

                return result;
            }

        }
        private void InitializeDatabasePath(string path)
        {
            _databasePath = new Lazy<string>(() => ResolveDatabasePath(path, "geolocation database"));
        }

        private void InitializeLocationsProvider(string path)
        {
            _locationsProvider = new Lazy<CsvGeographicLocationsProvider>(() => new CsvGeographicLocationsProvider(ResolveDatabasePath(path, "geographic locations database"), _options.LocationsCulture));
        }

        private string ResolveDatabasePath(string databasePath, string databaseType)
        {
            if (string.IsNullOrEmpty(databasePath))
            {
                _logger.Error($"No {databaseType} was specified. Some geolocation capabilities will be disabled.");
                return null;
            }

            if (databasePath.IndexOfAny(Path.GetInvalidPathChars()) >= 0)
            {
                _logger.Error($"The provided {databaseType} path '{databasePath}' contains one or many illegal characters. Some geolocation capabilities will be disabled.");
                return null;
            }

            var resolved = Environment.ExpandEnvironmentVariables(databasePath);
            if (!Path.IsPathRooted(resolved))
            {
                resolved = Path.GetFullPath(Path.Combine(_webHostingEnvironment.WebRootPath ?? AppDomain.CurrentDomain.BaseDirectory, resolved));
            }

            if (!File.Exists(resolved))
            {
                _logger.Error($"Unable to find a {databaseType} at the configured location '{databasePath}'. Some geolocation capabilities will be disabled.");
                return null;
            }

            return resolved;
        }

        private bool UseCityRequest(DatabaseReader reader)
        {
            return reader.Metadata.DatabaseType?.Contains("City") == true;
        }

        private static Capabilities GetCapabilitiesFromKnownDatabaseType(string databaseFile)
        {
            try
            {
                using (var reader = new DatabaseReader(databaseFile))
                {
                    switch (reader.Metadata.DatabaseType)
                    {
                        case "GeoLite2-City":
                        case "GeoIP2-City":
                            return Capabilities.ContinentCode | Capabilities.CountryCode | Capabilities.Region | Capabilities.Location;
                        case "GeoLite2-Country":
                        case "GeoIP2-Country":
                            return Capabilities.ContinentCode | Capabilities.CountryCode;
                    }
                }
            }
            catch (InvalidDatabaseException ex)
            {
                _logger.Error("Unable to read database type from geolocation database", ex);
            }

            return 0;
        }
    }
}
